import { NextRequest, NextResponse } from 'next/server';
import { getComprehensiveDatabase, initComprehensiveDatabase } from '@/src/lib/comprehensiveDatabase';
import { CustomerFormSchema } from '@/src/utils/comprehensiveValidators';

export async function GET(request: NextRequest, { params }: { params: { id: string } }) {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        const { id } = await params;
        const customer = db.prepare('SELECT * FROM customers WHERE id = ?').get(id);

        if (!customer) {
            return NextResponse.json({ error: 'Customer not found' }, { status: 404 });
        }

        return NextResponse.json(customer);
    } catch (error) {
        console.error('Error fetching customer:', error);
        return NextResponse.json({ error: 'Failed to fetch customer' }, { status: 500 });
    }
}

export async function PUT(request: NextRequest, { params }: { params: { id: string } }) {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        const { id } = await params;
        const body = await request.json();

        // Validate the request body
        const validatedData = CustomerFormSchema.parse(body);

        const now = new Date().toISOString();

        const result = db.prepare(`
            UPDATE customers 
            SET name = ?, email = ?, phone = ?, company = ?, status = ?, stage = ?, 
                assigned_sales_rep = ?, value = ?, notes = ?, updated_at = ?
            WHERE id = ?
        `).run(
            validatedData.name,
            validatedData.email || null,
            validatedData.phone || null,
            validatedData.company || null,
            validatedData.status,
            validatedData.stage,
            validatedData.assigned_sales_rep || null,
            validatedData.value,
            validatedData.notes || null,
            now,
            id
        );

        if (result.changes === 0) {
            return NextResponse.json({ error: 'Customer not found' }, { status: 404 });
        }

        const updatedCustomer = db.prepare('SELECT * FROM customers WHERE id = ?').get(id);
        return NextResponse.json(updatedCustomer);
    } catch (error) {
        console.error('Error updating customer:', error);
        if (error instanceof Error && error.name === 'ZodError') {
            return NextResponse.json({ error: 'Validation error', details: error.message }, { status: 400 });
        }
        return NextResponse.json({ error: 'Failed to update customer' }, { status: 500 });
    }
}

export async function DELETE(request: NextRequest, { params }: { params: { id: string } }) {
    try {
        await initComprehensiveDatabase();
        const db = getComprehensiveDatabase();

        const { id } = await params;
        const result = db.prepare('DELETE FROM customers WHERE id = ?').run(id);

        if (result.changes === 0) {
            return NextResponse.json({ error: 'Customer not found' }, { status: 404 });
        }

        return NextResponse.json({ message: 'Customer deleted successfully' });
    } catch (error) {
        console.error('Error deleting customer:', error);
        return NextResponse.json({ error: 'Failed to delete customer' }, { status: 500 });
    }
}
